function circos(names,data,threshold,projection,turn)
% CIRCOS(NAMES, DATA, THRESHOLD, PROJECTION, TURN)
% Draws a circos-like fig.
%
% Names     - a cell array of names (length n)
% data      - a matrix [n n], indicating linkages. Is assumed symmetrical.
% threshold - logical matrix of which connections to show. If omitted, is assumed to be all TRUE.
% projection (optional) - if given and not empty, only correlation with this element are shown
% turn (optional, default=0) - if given, turns the whole plot by this angle

% Oct 30 2014: Revisited. Projection implemented.
% Jan 06 2015: Usability changes (+TURN input argument)

if(nargin<4)
    projection = NaN;    
end
if(nargin<5)
    turn = 0;
end
if(isempty(projection))
    projection = NaN;
end


nPoints = 50;  % How many points per curve

lineWidthMin = 1;   % How thick should the thinnest lines be
lineWidthMax = 6;   % And up to how thick they can become
fontSize = 10;      % Usually 8 is OK
colorFading = 0.9;  % From 0 (no fading) up to 1 (strong fading)

rCirc           = 1.05;
rText           = 1.15;
rPoints         = 1.075;   % Points that indicate
rWhCoeff        = 0.5;      % Whiskers coeff (control points) R. How perpendicular should the arches be when sticking to the circle
coeffPointy     = 0.1;     % How pointy should the whiskers be. Increase to make more blunt.
curvature       = 1;      % 1 for neutral
rMidMin         = 0.95;        % Whether adjucent points whould still produce an arch. 1 for no arch; reduce to make it more "bendy"
alims            = 2.4;       % Limits for the axes

n = length(data);
if(nargin<3)
    threshold = ones(size(data));
end

dataMin = min(abs(data(threshold(:)==1)));
dataMax = max(abs(data(threshold(:)==1)));
dataSpread = dataMax-dataMin;

figure; hold on;
set(gca,'DataAspectRatio',[1 1 1],'XLim',[-1 1]*alims,'YLim',[-1 1]*alims);
hC = circle([0 0],rCirc,nPoints*2,'k-');
set(hC,'LineWidth',2);

for(q=1:n)  % Placing texts
    a = 2*pi/n*(q-1)+turn;                              % Angle
    plot(cos(a)*rPoints,sin(a)*rPoints,'k.');
    %plot(cos(a)*rFar,sin(a)*rFar,'w.','visible','on');
    aDeg = a/pi*180;                                    % "Text" seems to expect degrees as one of its parameters
    if((aDeg>90) & (aDeg<270))
        hT = text(cos(a)*rText,sin(a)*rText,[names{q}],'Rotation',aDeg-180,'HorizontalAlignment','right');
    else
        hT = text(cos(a)*rText,sin(a)*rText,[names{q}],'Rotation',aDeg,'HorizontalAlignment','left');
    end
    set(hT,'FontSize',fontSize);
end

todo = [];  % To Do Array
for(q=2:n)
    for(w=1:q-1)
        if(threshold(q,w))
            if(isnan(projection) || (projection==q) || (projection==w))     % Either no selected, or this one was selected
                todo = [todo; q w data(q,w)];
            end
        end
    end
end

[~,order] = sort(abs(todo(:,3)));                   % Sort by strength
todo = todo(order,:);

for(i=1:length(todo))
    q = todo(i,1);
    w = todo(i,2);
    aq = 2*pi/n*(q-1)+turn;                         % Angle for the 1st point
    aw = 2*pi/n*(w-1)+turn;
    am = (aq+aw)/2;             
    relang = abs(aq-aw)/pi;                         % Relative angle
    if(abs(aq-aw)>pi)
        am = am+pi;
        relang = 2-relang;
        inv = -1;                                   % Inversion flag for whiskers
    else
        inv = 1;                                    % No inversion
    end
    rmid = rMidMin*(1-relang)^curvature;            % How far from the center the arch chould pass.
    centralPoint = p(rmid,am);
    rWh = 1-(1-rmid)*rWhCoeff;
    whiskers = p(sqrt(rWh)*coeffPointy,am+pi/2);        
    knots = [p(1,aq); centralPoint; p(1,aw)];
    cp1   = [p(rWh,aq); centralPoint-whiskers*inv];
    cp2   = [centralPoint+whiskers*inv; p(rWh,aw)];
    h = displayBezier(knots,cp1,cp2,nPoints);            
    if(data(q,w)>0)
        myColor = [1        0       0 ];            % Kinda red
    else
        myColor = [0        0.2     1 ]*0.8;        % Kinda blue
    end
    if(isfinite(data(q,w)))
        k = (abs(data(q,w))-dataMin)/dataSpread;    % Importance coefficient, 0<k<1
        set(h,'LineWidth',lineWidthMin+k*(lineWidthMax-lineWidthMin),'Color',myColor+(1-myColor)*(1-k)*colorFading);
        %fprintf('%f\t%f\t%f\t%3d\n',dataMin,abs(data(q,w)),dataMax,round(k*100));
    end
end

set(gca,'XTick',[],'YTick',[],'Visible','off');
set(gcf,'Color','white');
hold off;

end

function h = displayBezier(knots,cp1,cp2,npoints,spec)
%Code subject: Simple display of a set of 2d cubic Bezier curve segments
%Programmer: Aaron Wetzler, aaronwetzler@gmail.com
%Date:12/12/2009
%
%displayBezier(knots,cp1,cp2)
%knots - Set of points with each row holding a point . In this case they must be 2-d. The code can
%easily be changed to display 3d points as well.
%[cp1,cp2]- the set of control points used to determine the characteristics
%of the Bezier segments
%
%Each segment will be drawn with 11 subsegments i.e. straight lines

if(nargin<5)
    spec = 'b-';
end

n=size(cp1,1);
dim=size(cp1,2);

t=[0:1/npoints:1]';
t=repmat(t,1,dim);
lt=size(t,1);

cpnts=cat(3, knots(1:end-1,:), cp1, cp2, knots(2:end,:));
hold on;
x = []; y = [];
for i=1:n
    B=repmat(cpnts(i,:,1),lt,1).*((1-t).^3) + 3*repmat(cpnts(i,:,2),lt,1).*(t.*(1-t).^2) + 3*repmat(cpnts(i,:,3),lt,1).*((t.^2).*(1-t))+repmat(cpnts(i,:,4),lt,1).*(t.^3);
    x = [x; B(:,1)];
    y = [y; B(:,2)];
    %plot(knots(:,1),knots(:,2),'.b');
end

%h = plot(B(:,1),B(:,2),spec);
h = plot(x,y,spec);
end

function v = p(r,a)
% Polar coordinates
v = [0 0];
v(1) = r*cos(a);
v(2) = r*sin(a);
end